from __future__ import absolute_import, print_function, unicode_literals
import Live
from ableton.v2.control_surface import ControlSurface, MIDI_NOTE_TYPE
from ableton.v2.control_surface.input_control_element import *
from ableton.v2.base import task
from.socket_manager import SocketManager
from bisect import bisect_left
from .mapping_manager import MappingComponent
from .setlist_manager import SetlistManager
import os
from collections import deque
import time
import re
import json

class ShowFlow(ControlSurface):
    def __init__(self, c_instance):
        super(ShowFlow, self).__init__(c_instance)
        with self.component_guard():
            self.log_path = os.path.join(os.path.expanduser("~"), "Music", "cue2live.log.txt")
            self.lk_path = os.path.join(os.path.expanduser("~"), "Music", "cue2live_license.json")
            try:
                with open(self.lk_path, 'r') as lk:
                    self.lk_active = True
            except FileNotFoundError:
                self.lk_active = False
            self.preferences_key = "Cue2LivePreferencesKey"
            _ = self.preferences  # Access to initialize
            self.socket_manager = SocketManager(self)
            self.start_socket_server()
            self.c_instance = c_instance  # Assign directly here
            self.song_cues = {}
            self.stop_cues = {}
            self.loop_starts = {}
            self.loop_ends = {}
            self.current_cue_mode = 1
            self.cue_modes = {1: "Jump-on-PB", 2: "Jump-on-select"}
            self.set_man = SetlistManager(self)
            self.initialize_cue_points()    # Initialize cue points tracking
            self.sorted_stops = sorted(self.stop_cues.values())
            self.sorted_loop_starts = sorted(self.loop_starts.values())
            self.sorted_loop_ends = sorted(self.loop_ends.values())
            self.current_song_cue = None
            self.set_man.recall_setlist()   # Recall setlist data
            self.song.add_cue_points_listener(self.on_cue_points_changed)  # Listen for changes
            self.song.add_current_song_time_listener(self.stop_check)
            self.is_playing = False
            self.song.add_is_playing_listener(self.send_is_playing)
            self.loop_on = True
            self.loop_length = 16
            self.loop_start = None
            self.loop_update = False
            self.map_comp = MappingComponent(self)# Pass the current instance (self) to MappingComponent
            self.pending_mappings = deque()  # Queue to store pending mappings
            self.restore_preferences(self.preferences)  # Restore preferences during initialization
            if self.stored_mode:
                self.current_cue_mode = self.stored_mode
            self._log("Preferences loaded: {}".format(self.preferences))
            self._log(self.map_comp.control_mappings)
            self._tasks.add(task.run(self.check_lk))
            self.socket_manager.server.server.listen(5)
            self.socket_manager.server._poll_socket()
            
        
    def add_map_task(self):
        self._tasks.add(task.run(self._process_mapping_task))
            
    def start_socket_server(self):
        self._log("Starting socket server...")
        self.socket_manager.start_server()

    def disconnect(self):
        # Stop the socket server when Ableton unloads the script
        self._log("Stopping socket server...")
        self.socket_manager.server.thread_running = False
        self.socket_manager.stop_server()
        self.store_preferences(self.preferences)  # Store preferences during disconnection

        # Call the parent class's disconnect method if applicable
        super(ShowFlow, self).disconnect()

    def store_preferences(self, preferences):
        """Store the preferences for all components."""
        self.map_comp.store(preferences)
#        self.socket_manager.store(preferences)
        self.preferences["stored_mode"] = self.current_cue_mode
        self._pre_serialize()  # Serialize preferences
        self._log("Preferences stored.")

    def restore_preferences(self, preferences):
        """Restore the preferences for all components."""
        self.map_comp.restore(preferences)
        self.stored_mode = preferences.get("stored_mode", {})
        self._log("Preferences restored.")

                
    def _process_mapping_task(self):
        """Processes mappings from the queue."""
        self._log("Processing mapping queue: {}".format(list(self.pending_mappings)))
        if len(self.pending_mappings) == 0:
            self._mapping_task.kill()
            self._log("No tasks left in queue; stopping task")
        else:
            mapping = self.pending_mappings.popleft()
            self._log("Dequeued mapping: {}".format(mapping))  # Debugging: log the dequeued mapping

            try:
                self.assign_control(
                    mapping["control_name"],
                    mapping["msg_type"],
                    mapping["channel"],
                    mapping["identifier"]
                )
                self._log("Successfully assigned control: {}".format(mapping['control_name']))  # Debugging: log successful assignment
            except Exception as e:
                self._log("Error in assigning control: {}, Error: {}".format(mapping['control_name'], e))  # Debugging: log errors
            

            
    def assign_control(self, control_name, msg_type, channel, identifier):
        """Assign a control dynamically."""
        if control_name not in self.map_comp.control_mappings:
            self._log("Unknown control name: {}".format(control_name))
            return

        # Remove the old listener if it exists
        if self.map_comp.control_mappings[control_name]:
            try:
                self.map_comp.control_mappings[control_name].remove_value_listener(getattr(self, "handle_{}".format(control_name)))
            except AttributeError:
                self._log("No value listener")

        # Create a new InputControlElement

        # Dynamically create a variable for the InputControlElement
        globals()[control_name] = InputControlElement(
            msg_type=msg_type,
            channel=channel,
            identifier=identifier,
        )

        # Add a listener to the control
        globals()[control_name].add_value_listener(
            getattr(self, "handle_{}".format(control_name))
        )
        self._log("Assigned {} control to msg_type={}, channel={}, identifier={}".format(control_name, msg_type, channel, identifier))

    def remove_control(self, control_name):
        """Remove a control mapping dynamically."""
        if control_name not in self.map_comp.control_mappings:
            self._log("Control name not found: {}".format(control_name))
            return

        # Remove the value listener if one exists
        control = self.map_comp.control_mappings[control_name]
        if control:
            try:
                self._log("Removing value listener for control: {}".format(control_name))
                globals()[control_name].remove_value_listener(getattr(self, "handle_{}".format(control_name)))
                globals()[control_name].disconnect()
            except AttributeError:
                self._log("No listener found for control: {}".format(control_name))

        # Set the mapping to None
        self.map_comp.control_mappings[control_name] = None
        self._log("Removed mapping for '{}'.".format(control_name))


    def handle_play(self, value):
        """Handles the play button by jumping to the current song's cue point and starting playback."""
        if value > 0:  # Button pressed
            if not self.set_man.setlist:
                return
            if self.current_cue_mode == 1:
                self.jump_to_and_play(self.current_song_cue)
            elif self.current_cue_mode == 2:
                if self.song.current_song_time == self.current_song_cue.time:
                    self.song.start_playing()
                else:
                    self.jump_to_and_play(self.current_song_cue)
        
    def handle_stop(self, value):
        """Handle stop button."""
        if value > 0:  # Button pressed
            self.song.stop_playing()

    def handle_next(self, value):
        """Handle next button."""
        if value > 0:  # Button pressed
            if not self.set_man.setlist:
                return
            self.set_man.setlist_index = self.set_man.setlist_index + 1
            self._log("Moved to next song: {}".format(self.set_man.setlist[self.set_man.setlist_index]))

    def handle_previous(self, value):
        """Handle previous button."""
        if value > 0:  # Button pressed
            if not self.set_man.setlist:
                return
            self.set_man.setlist_index = self.set_man.setlist_index - 1
            self._log("Moved to previous song: {}".format(self.set_man.setlist[self.set_man.setlist_index]))
            
    def _log(self, message):
        with open(self.log_path, "a") as f:
            f.write("{}\n".format(message))
    

    def initialize_cue_points(self):
        """Initializes the dictionary with existing cue points."""
        self.song_cues.clear()  # Ensure it's empty to start fresh
        for cue_point in self.song.cue_points:
            name = cue_point.name.strip().lower()

            if name == 'stop' or name == 'autostop':
                self.add_cue_stop(cue_point)

            elif re.match(r'^loop(end)?(\d+)?$', name):
                if name == 'loopend':
                    self.add_loop_end(cue_point)
                else:
                    self.add_loop_start(cue_point)

            else:
                self.add_cue_point(cue_point)

        self._log("Initialized song cues: {}\nInitialized stop_cues: {}".format(
            self.song_cues, list(self.stop_cues.values())
        ))

    def on_cue_points_changed(self):
        """Handles adding or removing cue points."""
        current_cues = set(id for id in self.song_cues)  # id is hash(cue_point) object
        current_loop_starts = set(self.loop_starts)  # cue hashes
        current_loop_ends = set(self.loop_ends)  # cue hashes
        current_stop_cues = set(id for id in self.stop_cues)  # id is hash(cue_point) object
        ableton_stop_cues = set()
        ableton_stop_cues_hash = set()
        ableton_cues = set()
        ableton_cues_hash = set()
        ableton_loop_start_hashes = set()
        ableton_loop_end_hashes = set()

        
        for cue in self.song.cue_points:
            name = cue.name.strip().lower()
            if name == 'stop' or name == 'autostop':
                ableton_stop_cues.add(cue)
                ableton_stop_cues_hash.add(hash(cue))
            elif re.match(r'^loop(end)?(\d+)?$', name):
                if name == "loopend":
                    ableton_loop_end_hashes.add(hash(cue))
                else:
                    ableton_loop_start_hashes.add(hash(cue))
            else:
                ableton_cues.add(cue)
                ableton_cues_hash.add(hash(cue))

        # Check for new cue points
        for cue_point in ableton_cues:
            if hash(cue_point) not in current_cues:
                self.add_cue_point(cue_point)

        # Check for deleted cue points
        for cue_id in current_cues:
            if cue_id not in ableton_cues_hash:
                self.remove_cue_point(cue_id)

        # Check for deleted stop cues
        for cue_id in current_stop_cues:
            if cue_id not in ableton_stop_cues_hash:
                self.remove_cue_stop(cue_id)
            
        for cue_hash in current_loop_starts:
            if cue_hash not in ableton_loop_start_hashes:
                self.remove_loop_start(cue_hash)

        for cue_hash in current_loop_ends:
            if cue_hash not in ableton_loop_end_hashes:
                self.remove_loop_end(cue_hash)
        
        # We don't need to address adding stops and loops here because when a cue is created
        # it immediately defaults to the number 1, which is treated as a normal cue.
        
                
    def add_cue_stop(self, cue_point):
        self.stop_cues[hash(cue_point)] = cue_point.time
        cue_point.add_name_listener(lambda cp=cue_point: self.update_cue_point_name(cp))
        cue_point.add_time_listener(lambda cp=cue_point: self.update_cue_point_time(cp))
        self.update_sorted_stops()
        self._log("Adding stop at {}".format(cue_point.time))
        
        
    def remove_cue_stop(self, cue_id):
        self._log("Removing stop at {}".format(self.stop_cues[cue_id]))
        del self.stop_cues[cue_id]
        self.update_sorted_stops()
        
    #New code
        
    def add_loop_start(self, cue_point):
        cue_hash = hash(cue_point)
        self.loop_starts[cue_hash] = cue_point.time
        self.update_sorted_loop_starts()
        cue_point.add_name_listener(lambda cp=cue_point: self.update_cue_point_name(cp))
        cue_point.add_time_listener(lambda cp=cue_point: self.update_cue_point_time(cp))
        self._log("Added Loop start at {}".format(cue_point.time))

    def remove_loop_start(self, cue_id):
        if cue_id in self.loop_starts:
            del self.loop_starts[cue_id]
            self.update_sorted_loop_starts()

    def add_loop_end(self, cue_point):
        cue_hash = hash(cue_point)
        self.loop_ends[cue_hash] = cue_point.time
        self.update_sorted_loop_ends()
        cue_point.add_name_listener(lambda cp=cue_point: self.update_cue_point_name(cp))
        cue_point.add_time_listener(lambda cp=cue_point: self.update_cue_point_time(cp))
        self._log("Added Loop end at {}".format(cue_point.time))

    def remove_loop_end(self, cue_id):
        if cue_id in self.loop_ends:
            del self.loop_ends[cue_id]
            self.update_sorted_loop_ends()

    
    #End New Code


    def add_cue_point(self, cue_point):
        """Adds a new cue point to the dictionary."""
        cue_name = cue_point.name.split('_', 1)
        song_name = cue_name[0]
        song_length = self.re_match(cue_name)
        self.song_cues[hash(cue_point)] = {
        "name": song_name,
        "time": cue_point.time,
        "length": song_length
        }
        cue_point.add_name_listener(lambda cp=cue_point: self.update_cue_point_name(cp))
        cue_point.add_time_listener(lambda cp=cue_point: self.update_cue_point_time(cp))
        self.set_man.refresh_song_pool()
        self._log("Added cue point: {}".format(song_name))
        
    def remove_cue_point(self, cue_id):
        """Removes a cue point from the dictionary."""
        if cue_id in self.song_cues:
            removed_cue = self.song_cues.pop(cue_id)
        self.set_man.refresh_song_pool()
        self._log("Removed cue point: {}".format(removed_cue["name"]))
        
    def parse_cue_metadata(self, cue_point):
        parts = cue_point.name.split('_', 1)
        return parts[0], self.re_match(parts)

    def update_cue_point_name(self, cue_point):
        """Updates internal dictionaries when a cue point name changes."""
        cue_hash = hash(cue_point)
        name = cue_point.name.strip().lower()

        def reassign_as_song_cue():
            song_name, song_length = self.parse_cue_metadata(cue_point)
            self.song_cues[cue_hash] = {
                "name": song_name,
                "time": cue_point.time,
                "length": song_length
            }
            self.set_man.refresh_song_pool()

        if cue_hash in self.song_cues:
            if name in ('stop', 'autostop'):
                self.remove_cue_point(cue_hash)
                self.stop_cues[cue_hash] = cue_point.time
                self.update_sorted_stops()
            elif re.match(r'^loop(end)?(\d+)?$', name):
                self.remove_cue_point(cue_hash)
                if name == 'loopend':
                    self.add_loop_end(cue_point)
                else:
                    self.add_loop_start(cue_point)
            else:
                song_name, song_length = self.parse_cue_metadata(cue_point)
                self.song_cues[cue_hash]["name"] = song_name
                self.song_cues[cue_hash]["length"] = song_length
                self.set_man.refresh_song_pool()

        elif cue_hash in self.stop_cues and name not in ('stop', 'autostop'):
            del self.stop_cues[cue_hash]
            self.update_sorted_stops()
            reassign_as_song_cue()

        elif cue_hash in self.loop_starts:
            self.remove_loop_start(cue_hash)
            if name == 'loopend':
                self.add_loop_end(cue_point)
            elif re.match(r'^loop(end)?(\d+)?$', name):
                self.add_loop_start(cue_point)
            else:
                reassign_as_song_cue()

        elif cue_hash in self.loop_ends:
            self.remove_loop_end(cue_hash)
            if name == 'loopend':
                self.add_loop_end(cue_point)
            elif re.match(r'^loop(end)?(\d+)?$', name):
                self.add_loop_start(cue_point)
            else:
                reassign_as_song_cue()

 
    def update_cue_point_time(self, cue_point):
        cue_hash = hash(cue_point)
        name = cue_point.name.strip().lower()

        if name in ('stop', 'autostop'):
            self.stop_cues[cue_hash] = cue_point.time
            self.update_sorted_stops()

        elif cue_hash in self.song_cues:
            self.song_cues[cue_hash]["time"] = cue_point.time
            self.set_man.refresh_song_pool()

        elif cue_hash in self.loop_starts:
            self.loop_starts[cue_hash] = cue_point.time
            self.update_sorted_loop_starts()

        elif cue_hash in self.loop_ends:
            self.loop_ends[cue_hash] = cue_point.time
            self.update_sorted_loop_ends()

            
    def update_sorted_stops(self):
        try:
            self.sorted_stops = sorted(self.stop_cues.values())
        except AttributeError:
            return
    
    def update_sorted_loop_starts(self):
        try:
            self.sorted_loop_starts = sorted(self.loop_starts.values())
            self._log("Sorted Loop Starts: {}".format(self.sorted_loop_starts))
        except AttributeError:
            return
                
    def update_sorted_loop_ends(self):
        try:
            self.sorted_loop_ends = sorted(self.loop_ends.values())
            self._log("Sorted Loop Ends: {}".format(self.sorted_loop_ends))
        except AttributeError:
            return
    
                
    def selected_cue(self, song_id):
        self._log("selecting cue")
        for cue in self.song.cue_points:
            if song_id == hash(cue):
                self.current_song_cue = cue
                if self.current_cue_mode == 2:
                    if self.song.is_playing:
                        self.song.current_song_time = cue.time
                    else:
                        cue.jump()
                self._log("Current song cue changed to {}".format(self.song_cues[song_id]))
                break

    def send_is_playing(self):
        """Sends is playing update to clients if change in playing status is triggered by client action"""
        self.socket_manager.server.send_is_playing(self.song.is_playing)

    def jump_to_and_play(self, cue_to_play):
        """Jumps to the specified time in seconds and starts playback."""
        if not self.song.is_playing:
            cue_to_play.jump()
            self.song.start_playing()
        else:
            cue_to_play.jump()
            


    def stop_check(self):
        if self.song.is_playing:
            current_time = self.song.current_song_time
            for stop_time in self.sorted_stops:
                if current_time <= stop_time + 1:
                    if stop_time <= current_time <= stop_time + 0.1:
                        self.song.stop_playing()
                        return
                # --- LOOP check ---
            for loop_time in self.sorted_loop_starts:
                if current_time <= loop_time + 1:
                    if loop_time <= current_time <= loop_time + 0.1:
                        self.prepare_loop_update(loop_time)
                        return
                        
    def prepare_loop_update(self, loop_time):
        try:
            import builtins
            bnext = builtins.next
            # Reverse lookup: find cue_hash where value matches loop_time
            cue_hash = bnext((k for k, v in self.loop_starts.items() if abs(v - loop_time) < 0.01), None)

            if cue_hash is None:
                return

            cue_point = bnext((cp for cp in self.song.cue_points if hash(cp) == cue_hash), None)

            if cue_point is None:
                return

            name = cue_point.name.strip().lower()
            match = re.match(r'loop(\d+)?', name)

            if match and match.group(1):  # loop8, loop16, etc.
                loop_length = int(match.group(1)) * 4
            else:
                # try to find a matching LOOPEND
                loop_end_candidates = [v for v in self.loop_ends.values() if v > loop_time]
                if loop_end_candidates:
                    loop_length = loop_end_candidates[0] - loop_time
                else:
                    loop_length = 16.0

            self.loop_start = loop_time
            self.loop_length = loop_length
            self.loop_update = True

        except Exception as e:
            self._log("Exception in prepare_loop_update: {}".format(e))



                        
    def update_loop(self):
        if self.loop_update:
            if not self.song.loop:
                self.song.loop = True
            self.song.loop_start = self.loop_start
            self.song.loop_length = self.loop_length
            self.loop_update = False
            return


    def re_match(self, cue_name):
        if len(cue_name) > 1:
            song_length_candidate = cue_name[1].strip()  # Remove leading and trailing spaces
            # Regex for "m:ss" OR "mm:ss" OR "hh:mm:ss"
            if re.match(r"^\d{1,2}:\d{2}(:\d{2})?$", song_length_candidate):
                song_length = song_length_candidate  # Valid time format
            else:
                song_length = ""  # Invalid format, default to empty
        else:
            song_length = ""  # No second part
        return song_length
        
    def check_lk(self):
        if not self.lk_active:
            if self.song.is_playing:
                self.song.stop_playing()
            self._tasks.add(task.sequence(task.WaitTask(30.0), task.run(self.check_lk)))
        else:
            self._log("License Validated!")
            return
